import * as state from './state.js';
import * as ui from './ui.js';
import * as ipc from './ipc.js';
import * as input from './input.js';
import * as utils from './utils.js';

async function appLogic() {
    
    const loadedSettings = await ipc.getSettings();
    state.setInitialSettings(loadedSettings);

    
    const savedPlatformId = state.appSettings.lastPlatformId;
    let targetPlatform = state.PLATFORMS.GENESIS; // Default
    if (savedPlatformId) {
        const found = Object.values(state.PLATFORMS).find(p => p.id === savedPlatformId);
        if (found) targetPlatform = found;
    }
    state.setCurrentPlatform(targetPlatform);

    
    ui.updateMainLogo();

    
    const initialData = await ipc.getInitialData(targetPlatform.id);
    if (!initialData) {
        throw new Error("Failed to receive initial data from main process.");
    }
    state.setInitialData(initialData);

    
    
    const translations = await ipc.loadTranslations(state.appSettings.language);
    utils.applyTranslationsToDOM(translations);

    if (state.appSettings.bgmPath === "off") {
        const offIndex = state.availableBgmTracks.findIndex(track => track.name === 'Music Off');
        state.appSettings.currentBgmIndex = offIndex !== -1 ? offIndex : 0;
    } else {
        let trackToPlay = state.availableBgmTracks.find(track => track.path === state.appSettings.bgmPath || (track.name === 'Default' && state.appSettings.bgmPath === 'default'));
        if (trackToPlay && trackToPlay.path) {
            const savedIndex = state.availableBgmTracks.findIndex(track => track.path === trackToPlay.path);
            state.appSettings.currentBgmIndex = savedIndex !== -1 ? savedIndex : 1;
            utils.backgroundMusic.src = `file:///${trackToPlay.path.replace(/\\/g, '/')}`;
            utils.backgroundMusic.play().catch(e => {});
        } else {
            const offIndex = state.availableBgmTracks.findIndex(track => track.name === 'Music Off');
            state.appSettings.currentBgmIndex = offIndex !== -1 ? offIndex : 0;
        }
    }
    
    utils.backgroundMusic.loop = true;
    utils.backgroundMusic.volume = state.appSettings.bgmVolumeLevel / 10 * 0.5;
    utils.navigateSound.src = state.UI_URLS.navigateSoundUrl; utils.navigateSound.volume = 0.7;
    utils.confirmSound.src = state.UI_URLS.confirmSoundUrl;
    utils.cancelSound.src = state.UI_URLS.cancelSoundUrl;

    await ui.renderUI();

    if (state.games.length > 0) {
        
        const initialIndex = Math.max(0, Math.min(state.games.length - 1, state.appSettings.lastGameIndex || 0));
        ui.updateSelection(initialIndex, false, true);
    } else {
        ui.gameTitleElement.textContent = utils.t('game_title_no_games');
    }

    ui.init({ onProcessStarted: input.resetInactivityTimer });
    ipc.init({ onProcessEnded: input.resetInactivityTimer });
    input.initializeInput();
    input.resetInactivityTimer();
}

async function initialize() {
    try {
        const introOverlay = document.getElementById('intro-overlay');
        const selectionContainer = document.getElementById('selection-container');
        const actionsPanel = document.getElementById('actions-panel');
        const wait = ms => new Promise(resolve => setTimeout(resolve, ms));

        introOverlay.classList.add('visible');
        await wait(2000);
        introOverlay.classList.remove('visible');
        await wait(200);

        await appLogic();
        
        window.addEventListener('beforeunload', () => {
            ipc.saveSettingsOnExit(state.appSettings);
        });

        selectionContainer.classList.add('visible');
        actionsPanel.classList.add('visible');

    } catch (error) {
        console.error("Renderer: A fatal error occurred during initialization:", error);
        document.body.innerHTML = `<div style="color: red; font-family: monospace; font-size: 16px; padding: 20px; background-color: #111; height: 100vh;">
            <h1>Application Failed to Load</h1><p>A critical error occurred. Please check the developer console (Ctrl+Shift+I) for more details.</p>
            <pre style="color: #ff8080;">${error.stack}</pre></div>`;
    }
}

initialize();